import pytest
import json
import numpy as np
from pathlib import Path

# Assuming your svd_components.py is in python_src/dsvd_module/
# Adjust the import path if your project structure is different
# This import assumes that pytest is run from the project root directory ("PortPrj")
# and "PortPrj" is in the PYTHONPATH or python_src is recognizable as a package.
from python_src.dsvdc import drot1

# --- Helper function to load test data ---
def load_drot1_test_data():
    """Loads test data for drot1 from the specified JSON file."""
    # Construct the path to the JSON file relative to this test file's location
    # Assumes:
    # Project_Root/
    #  ├── tests/
    #  │   └── test_svd_components.py  (this file)
    #  └── test_data/
    #      └── golden_master_python/
    #          └── drot1/
    #              └── drot1_with_t_svd.json
    try:
        test_file_dir = Path(__file__).parent
        project_root = test_file_dir.parent # Assumes tests/ is one level down
        json_file_path = project_root / "test_data" / "golden_master_python" / "drot1" / "drot1_with_t_svd.json"
        
        with open(json_file_path, 'r') as f:
            test_cases = json.load(f)
        return test_cases
    except FileNotFoundError:
        pytest.fail(f"Test data file not found at {json_file_path}. "
                    "Please ensure the path is correct relative to your project structure "
                    "and where pytest is being run.")
    except json.JSONDecodeError:
        pytest.fail(f"Error decoding JSON from {json_file_path}. Please check the file format.")

# --- Parametrized test function ---
@pytest.mark.parametrize("test_case_data", load_drot1_test_data())
def test_drot1_against_golden_master(test_case_data):
    """
    Tests the Python drot1 function against a single test case from the 
    Golden Master JSON data.
    """
    call_id = test_case_data.get("call_id", "N/A")
    inputs = test_case_data["inputs"]
    expected_outputs = test_case_data["outputs"]

    # Prepare inputs for the drot1 function
    n = inputs["N_arg"]
    c = inputs["C_val"]
    s = inputs["S_val"]
    
    # Convert input arrays to NumPy arrays.
    # drot1 modifies dx and dy in-place, so these will be the actual results.
    # Use float64 to match Fortran's REAL(dp)
    dx_actual = np.array(inputs["DX_arr_IN"], dtype=np.float64)
    dy_actual = np.array(inputs["DY_arr_IN"], dtype=np.float64)

    # Prepare expected output arrays
    dx_expected = np.array(expected_outputs["DX_arr_OUT"], dtype=np.float64)
    dy_expected = np.array(expected_outputs["DY_arr_OUT"], dtype=np.float64)

    # Call the Fortran-ported Python function
    drot1(n, dx_actual, dy_actual, c, s)

    # Perform assertions using numpy.testing.assert_allclose for float comparisons
    # rtol (relative tolerance) and atol (absolute tolerance) can be adjusted
    # if needed, but these are common defaults.
    rtol = 1e-7  # Relative tolerance
    atol = 1e-9  # Absolute tolerance (useful for numbers close to zero)

    try:
        np.testing.assert_allclose(
            dx_actual, 
            dx_expected, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"DX array mismatch for call_id={call_id}"
        )
    except AssertionError as e:
        print(f"Failed DX comparison for call_id={call_id}")
        print(f"Actual DX: {dx_actual}")
        print(f"Expected DX: {dx_expected}")
        print(f"Difference: {dx_actual - dx_expected}")
        raise e

    try:
        np.testing.assert_allclose(
            dy_actual, 
            dy_expected, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"DY array mismatch for call_id={call_id}"
        )
    except AssertionError as e:
        print(f"Failed DY comparison for call_id={call_id}")
        print(f"Actual DY: {dy_actual}")
        print(f"Expected DY: {dy_expected}")
        print(f"Difference: {dy_actual - dy_expected}")
        raise e


# python_src.dsvdc から dswap1 をインポート
from python_src.dsvdc import dswap1 # dswap1 を追加

# --- dswap1 用のテストデータ読み込みヘルパー関数 ---
def load_dswap1_test_data():
    """dswap1 のテストデータを指定のJSONファイルからロードします。"""
    try:
        test_file_dir = Path(__file__).parent
        project_root = test_file_dir.parent 
        json_file_path = project_root / "test_data" / "golden_master_python" / "dswap1" / "dswap1_golden_master.json"
        
        with open(json_file_path, 'r') as f:
            test_cases = json.load(f)
        return test_cases
    except FileNotFoundError:
        pytest.fail(f"dswap1 のテストデータファイルが見つかりません: {json_file_path}")
    except json.JSONDecodeError:
        pytest.fail(f"JSONデコードエラー ({json_file_path})。ファイル形式を確認してください。")

# --- dswap1 用のパラメタライズドテスト関数 ---
@pytest.mark.parametrize("test_case_data", load_dswap1_test_data())
def test_dswap1_against_golden_master(test_case_data):
    """
    Python版 dswap1 関数を Golden Master JSON データの一つのテストケースに対してテストします。
    """
    call_id = test_case_data.get("call_id", "N/A")
    inputs = test_case_data["inputs"]
    expected_outputs = test_case_data["outputs"]

    # dswap1 関数のための入力を準備
    n = inputs["N_arg"]
    
    # 入力配列をNumPy配列に変換します。
    # dswap1 は dx と dy を直接変更するため、これらが実際の実行結果となります。
    # FortranのREAL(dp)に合わせてfloat64を使用します。
    dx_actual = np.array(inputs["DX_arr_IN"], dtype=np.float64)
    dy_actual = np.array(inputs["DY_arr_IN"], dtype=np.float64)

    # 期待される出力配列を準備
    dx_expected = np.array(expected_outputs["DX_arr_OUT"], dtype=np.float64)
    dy_expected = np.array(expected_outputs["DY_arr_OUT"], dtype=np.float64)

    # Pythonに移植された関数を呼び出し
    dswap1(n, dx_actual, dy_actual)

    # アサーション: numpy.testing.assert_allclose を使用して浮動小数点数を比較します。
    rtol = 1e-7  # 相対許容誤差
    atol = 1e-9  # 絶対許容誤差

    try:
        np.testing.assert_allclose(
            dx_actual, 
            dx_expected, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"dswap1: DX 配列が一致しません (call_id={call_id})"
        )
    except AssertionError as e:
        print(f"dswap1: DX 比較失敗 (call_id={call_id})")
        print(f"Actual DX: {dx_actual}")
        print(f"Expected DX: {dx_expected}")
        print(f"Difference: {dx_actual - dx_expected}")
        raise e

    try:
        np.testing.assert_allclose(
            dy_actual, 
            dy_expected, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"dswap1: DY 配列が一致しません (call_id={call_id})"
        )
    except AssertionError as e:
        print(f"dswap1: DY 比較失敗 (call_id={call_id})")
        print(f"Actual DY: {dy_actual}")
        print(f"Expected DY: {dy_expected}")
        print(f"Difference: {dy_actual - dy_expected}")
        raise e



from python_src.dsvdc import drotg # drotg を追加

# --- drotg 用のテストデータ読み込みヘルパー関数 ---
def load_drotg_test_data():
    """drotg のテストデータを指定のJSONファイルからロードします。"""
    try:
        test_file_dir = Path(__file__).parent
        project_root = test_file_dir.parent 
        json_file_path = project_root / "test_data" / "golden_master_python" / "drotg" / "drotg_golden_master.json"
        
        with open(json_file_path, 'r') as f:
            test_cases = json.load(f)
        return test_cases
    except FileNotFoundError:
        pytest.fail(f"drotg のテストデータファイルが見つかりません: {json_file_path}")
    except json.JSONDecodeError:
        pytest.fail(f"JSONデコードエラー ({json_file_path})。ファイル形式を確認してください。")

# --- drotg 用のパラメタライズドテスト関数 ---
@pytest.mark.parametrize("test_case_data", load_drotg_test_data())
def test_drotg_against_golden_master(test_case_data):
    """
    Python版 drotg 関数を Golden Master JSON データの一つのテストケースに対してテストします。
    """
    call_id = test_case_data.get("call_id", "N/A")
    inputs = test_case_data["inputs"]
    expected_outputs = test_case_data["outputs"]

    # drotg 関数のための入力を準備
    da_in = float(inputs["DA_val_IN"])
    db_in = float(inputs["DB_val_IN"])
    
    # Pythonに移植された関数を呼び出し
    actual_da, actual_db, actual_dc, actual_ds = drotg(da_in, db_in)

    # 期待される出力値を準備
    expected_da = float(expected_outputs["DA_val_OUT"])
    expected_db = float(expected_outputs["DB_val_OUT"])
    expected_dc = float(expected_outputs["DC_val_OUT"])
    expected_ds = float(expected_outputs["DS_val_OUT"])

    # アサーション: numpy.testing.assert_allclose を使用して浮動小数点数を比較します。
    rtol = 1e-7  # 相対許容誤差
    atol = 1e-9  # 絶対許容誤差 (ゼロに近い値の場合に有用)

    detailed_error_message = []

    try:
        np.testing.assert_allclose(
            actual_da, 
            expected_da, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"drotg: DA_OUT が一致しません (call_id={call_id})"
        )
    except AssertionError as e:
        detailed_error_message.append(str(e))
        detailed_error_message.append(f"  Actual DA: {actual_da}, Expected DA: {expected_da}, Diff: {actual_da - expected_da}")


    try:
        np.testing.assert_allclose(
            actual_db, 
            expected_db, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"drotg: DB_OUT が一致しません (call_id={call_id})"
        )
    except AssertionError as e:
        detailed_error_message.append(str(e))
        detailed_error_message.append(f"  Actual DB: {actual_db}, Expected DB: {expected_db}, Diff: {actual_db - expected_db}")

    try:
        np.testing.assert_allclose(
            actual_dc, 
            expected_dc, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"drotg: DC_val_OUT が一致しません (call_id={call_id})"
        )
    except AssertionError as e:
        detailed_error_message.append(str(e))
        detailed_error_message.append(f"  Actual DC: {actual_dc}, Expected DC: {expected_dc}, Diff: {actual_dc - expected_dc}")

    try:
        np.testing.assert_allclose(
            actual_ds, 
            expected_ds, 
            rtol=rtol, 
            atol=atol,
            err_msg=f"drotg: DS_val_OUT が一致しません (call_id={call_id})"
        )
    except AssertionError as e:
        detailed_error_message.append(str(e))
        detailed_error_message.append(f"  Actual DS: {actual_ds}, Expected DS: {expected_ds}, Diff: {actual_ds - expected_ds}")

    if detailed_error_message:
        pytest.fail("\n".join(detailed_error_message))

# (Ensure this import is at the top of tests/test_svd_components.py if not already for dsvdc)
# from python_src.dsvdc import dsvdc # Corrected name from previous thought process
from python_src.dsvdc import dsvdc

# --- dsvdc 用のテストデータ読み込みヘルパー関数 ---
def load_dsvdc_test_data():
    """dsvdc のテストデータを指定のJSONファイルからロードします。"""
    try:
        test_file_dir = Path(__file__).parent
        project_root = test_file_dir.parent  
        json_file_path = project_root / "test_data" / "golden_master_python" / "dsvdc" / "dsvdc_with_advanced_tester.json" #
        
        with open(json_file_path, 'r') as f:
            test_cases = json.load(f)
        return test_cases
    except FileNotFoundError:
        pytest.fail(f"dsvdc のテストデータファイルが見つかりません: {json_file_path}")
    except json.JSONDecodeError:
        pytest.fail(f"JSONデコードエラー ({json_file_path})。ファイル形式を確認してください。")

# --- dsvdc 用のパラメタライズドテスト関数 ---
@pytest.mark.parametrize("test_case_data", load_dsvdc_test_data())
def test_dsvdc_against_golden_master(test_case_data):
    """
    Python版 dsvdc 関数を Golden Master JSON データの一つのテストケースに対してテストします。
    """
    call_id = test_case_data.get("call_id", "N/A") # Assuming call_id might be added to JSON later
    inputs = test_case_data["inputs"] #
    expected_outputs = test_case_data["outputs"] #

    # --- 1. Prepare Inputs for dsvdc ---
    n_in = inputs["n"] #
    p_in = inputs["p"] #
    job_in = inputs["job"] #
    
    # Input matrix x. DSVDC modifies x.
    # The JSON provides x as list of lists, convert to NumPy array.
    # Assuming x is N x P.
    x_actual = np.array(inputs["x"], dtype=np.float64) #
    # Ensure x_actual has Fortran-like column-major order if dsvdc's logic subtly depends on it
    # when ported from Fortran, though direct element access and NumPy ops are often order-agnostic.
    # For this test, we'll assume standard NumPy C-order is fine unless issues arise.
    # If your dsvdc was written to expect x as (LDX, P) and use N from that, adjust accordingly.
    # Here, we assume x is passed as N x P.

    # --- 2. Prepare Output Arrays (to be modified by dsvdc) ---
    # s: Singular values. DSVDC documentation states s is MIN(N,P).
    # The JSON provides "s": null, so we can't directly compare its contents.
    # We'll create it, let dsvdc populate it, and note that comparison is skipped.
    m_calc_for_s_size = min(p_in, n_in + 1) # Calculate m as it's done in dsvdc for s array sizing
    s_actual = np.zeros(m_calc_for_s_size, dtype=np.float64)

    # e: Super-diagonal elements. DSVDC documentation states e is P.
    e_actual = np.zeros(p_in, dtype=np.float64)

    # u: Left singular vectors. Dimensions depend on job_in.
    jobu = (job_in % 100) // 10
    wantu = (jobu != 0)
    ncu_actual_cols = n_in # Default for jobu=0 or jobu=1 (full N x N U in principle for jobu=1)
    if jobu > 1: # jobu = 2 for N x min(N,P) U
        ncu_actual_cols = min(n_in, p_in)
    
    u_actual = None
    if wantu:
        # Allocate based on the shape of "u" in expected_outputs if available,
        # otherwise use n_in x ncu_actual_cols.
        # This ensures we allocate what the Fortran output (and thus dsvdc) produced.
        if "u" in expected_outputs and expected_outputs["u"] is not None:
            expected_u_shape = np.array(expected_outputs["u"]).shape #
            u_actual = np.zeros(expected_u_shape, dtype=np.float64)
        elif jobu == 1: # Expects N x N for full U from dsvdc's ncu logic for jobu=1
             u_actual = np.zeros((n_in, n_in), dtype=np.float64)
        elif jobu > 1: # Expects N x min(N,P)
             u_actual = np.zeros((n_in, min(n_in,p_in)), dtype=np.float64)


    # v: Right singular vectors. Dimensions depend on job_in. V is P x P if computed.
    wantv = ((job_in % 10) != 0)
    v_actual = None
    if wantv:
        # Allocate based on the shape of "v" in expected_outputs if available.
        if "v" in expected_outputs and expected_outputs["v"] is not None:
            expected_v_shape = np.array(expected_outputs["v"]).shape #
            v_actual = np.zeros(expected_v_shape, dtype=np.float64)
        else: # Default if not in JSON but wantv is true (should usually be in JSON if wantv)
            v_actual = np.zeros((p_in, p_in), dtype=np.float64)


    # --- 3. Call the Python dsvdc function ---
    # Note: dsvdc in svd_components.py is defined as:
    # def dsvdc(x, n, p, s, e, u, v, job):
    info_actual = dsvdc(x_actual, n_in, p_in, s_actual, e_actual, u_actual, v_actual, job_in)

    # --- 4. Prepare Expected Outputs from JSON ---
    info_expected = expected_outputs["info"] #
    
    e_expected = None
    if "e" in expected_outputs and expected_outputs["e"] is not None:
        e_expected = np.array(expected_outputs["e"], dtype=np.float64) #

    u_expected = None
    if wantu and "u" in expected_outputs and expected_outputs["u"] is not None:
        u_expected = np.array(expected_outputs["u"], dtype=np.float64) #
        # If Fortran order matters for comparison and JSON is column-major stored as row-major list-of-list:
        # u_expected = np.array(expected_outputs["u"], dtype=np.float64).T.copy(order='C').T # Or handle order in assert

    v_expected = None
    if wantv and "v" in expected_outputs and expected_outputs["v"] is not None:
        v_expected = np.array(expected_outputs["v"], dtype=np.float64) #

    # --- 5. Perform Assertions ---
    rtol = 1e-7 
    atol = 1e-9 
    detailed_error_message = []

    # Test info
    try:
        assert info_actual == info_expected, f"INFO flag mismatch (call_id={call_id}). Actual: {info_actual}, Expected: {info_expected}"
    except AssertionError as e:
        detailed_error_message.append(str(e))

    # Test e
    if e_expected is not None:
        try:
            # e_actual is p_in. e_expected from JSON might be shorter if p_in > min(n_in,p_in)-1 for some cases.
            # Compare only the relevant part if lengths differ, or ensure JSON matches expected length for e_actual.
            # DSVDC's e array has p elements. The meaningful superdiagonals are fewer.
            # For simplicity, assume e_expected covers the part of e_actual we care about.
            # Usually e has min(N,P)-1 meaningful non-zero elements, but is dimensioned P.
            # Let's compare up to the length of e_expected.
            len_to_compare_e = len(e_expected)
            np.testing.assert_allclose(e_actual[:len_to_compare_e], e_expected, rtol=rtol, atol=atol, err_msg=f"E array mismatch (call_id={call_id})")
        except AssertionError as e:
            detailed_error_message.append(f"E comparison failure: {str(e)}")
            detailed_error_message.append(f"  Actual E (first {len(e_expected)} els): {e_actual[:len(e_expected)]}")
            detailed_error_message.append(f"  Expected E: {e_expected}")

    # Test s (Commented out due to "s": null in JSON)
    # if "s" in expected_outputs and expected_outputs["s"] is not None:
    #     s_expected = np.array(expected_outputs["s"], dtype=np.float64)
    #     try:
    #         np.testing.assert_allclose(s_actual, s_expected, rtol=rtol, atol=atol, err_msg=f"S array mismatch (call_id={call_id})")
    #     except AssertionError as e:
    #         detailed_error_message.append(str(e))
    # else:
    #     detailed_error_message.append(f"Note: S array comparison skipped as expected 's' is null or missing in JSON for call_id={call_id}. Actual s computed: {s_actual}")
    # For now, just print a note if s was expected to be non-null but wasn't in JSON, or simply acknowledge it's populated.
    if not ("s" in expected_outputs and expected_outputs["s"] is not None):
         print(f"FYI for call_id={call_id}: S singular values computed by dsvdc are {s_actual}. Cannot compare with JSON as expected 's' is null or missing.")


    # Test u
    if wantu and u_expected is not None and u_actual is not None:
        try:
            # Ensure shapes are compatible for comparison. u_actual was allocated based on expected_u_shape.
            np.testing.assert_allclose(u_actual, u_expected, rtol=rtol, atol=atol, err_msg=f"U array mismatch (call_id={call_id})")
        except AssertionError as e:
            detailed_error_message.append(f"U comparison failure: {str(e)}")
            detailed_error_message.append(f"  Actual U (shape {u_actual.shape}):\n{u_actual}")
            detailed_error_message.append(f"  Expected U (shape {u_expected.shape}):\n{u_expected}")
    elif wantu and u_expected is None:
        detailed_error_message.append(f"Warning: wantu is True but U not found in expected_outputs for call_id={call_id}. U_actual shape: {u_actual.shape if u_actual is not None else 'None'}")


    # Test v
    if wantv and v_expected is not None and v_actual is not None:
        try:
            np.testing.assert_allclose(v_actual, v_expected, rtol=rtol, atol=atol, err_msg=f"V array mismatch (call_id={call_id})")
        except AssertionError as e:
            detailed_error_message.append(f"V comparison failure: {str(e)}")
            detailed_error_message.append(f"  Actual V (shape {v_actual.shape}):\n{v_actual}")
            detailed_error_message.append(f"  Expected V (shape {v_expected.shape}):\n{v_expected}")
    elif wantv and v_expected is None:
         detailed_error_message.append(f"Warning: wantv is True but V not found in expected_outputs for call_id={call_id}. V_actual shape: {v_actual.shape if v_actual is not None else 'None'}")


    # Test final state of x (Commented out: Not available in current JSON expected_outputs)
    # if "X_arr_OUT" in expected_outputs and expected_outputs["X_arr_OUT"] is not None:
    #     x_expected_out = np.array(expected_outputs["X_arr_OUT"], dtype=np.float64)
    #     try:
    #         np.testing.assert_allclose(x_actual, x_expected_out, rtol=rtol, atol=atol, err_msg=f"X output array mismatch (call_id={call_id})")
    #     except AssertionError as e:
    #         detailed_error_message.append(str(e))
    # else:
    #     # detailed_error_message.append(f"Note: Final X array comparison skipped as expected 'X_arr_OUT' is not in JSON for call_id={call_id}.")
    #     pass


    if detailed_error_message:
        # Join all collected error messages and fail the test
        pytest.fail("\n".join(detailed_error_message), pytrace=False)