# run_all_advanced_testers.py
#
# DESCRIPTION:
#   Runs a suite of advanced correctness tests for various SVD implementations.
#   This script executes each tester, captures its output, checks for errors,
#   and reports a summary of the results.
#
# USAGE:
#   python run_all_advanced_testers.py
#
# BEHAVIOR:
#   - A 'logs' directory will be created if it doesn't exist.
#   - The full stdout of each test run is saved to a corresponding file
#     in the 'logs' directory (e.g., 'advanced_tester_python_base.log').
#   - A summary table is printed to the console showing the PASS/FAIL
#     status for each test.

import sys
import os
import subprocess
import platform

# --- Configuration ---

# Absolute path to the MSYS2/MinGW-w64 Python required for the f2py-built module.
# This ensures that the correct environment and libraries are used for the f2py tester.
MSYS_PYTHON = r"D:\msys64\mingw64\bin\python.exe"


# Define the list of tester targets. Each dictionary specifies the display
# name, the command to run, and the log file for its output.
TESTER_TARGETS = [
    {
        "name": "Python (Base)",
        "command": [sys.executable, os.path.join("advanced_testers", "advanced_tester_python_base.py")],
        "log_file": "advanced_tester_python_base.log"
    },
    {
        "name": "Python (Vectorized)",
        "command": [sys.executable, os.path.join("advanced_testers", "advanced_tester_python_vectorized.py")],
        "log_file": "advanced_tester_python_vectorized.log"
    },
    {
        "name": "Python (Base+JIT)",
        "command": [sys.executable, os.path.join("advanced_testers", "advanced_tester_python_base_jit.py")],
        "log_file": "advanced_tester_python_base_jit.log"
    },
    {
        "name": "Python (Base+JIT-Cached)",
        "command": [sys.executable, os.path.join("advanced_testers", "advanced_tester_python_base_jit_cached.py")],
        "log_file": "advanced_tester_python_base_jit_cached.log"
    },
    {
        "name": "Python (f2py)",
        # Use the specific MSYS_PYTHON interpreter for the f2py test
#        "command": [MSYS_PYTHON, os.path.join("advanced_testers", "advanced_tester_python_f2py.py")],
        "command": [sys.executable, os.path.join("advanced_testers", "advanced_tester_python_f2py.py")],
        "log_file": "advanced_tester_python_f2py.log"
    },
    {
        "name": "Python (f2py_intent_out)",
        "command": [sys.executable, os.path.join("advanced_testers", "advanced_tester_python_f2py_intent_out.py")],
        "log_file": "advanced_tester_python_f2py_intent_out.log"
    },
    {
        "name": "Python (ctypes)",
        "command": [sys.executable, os.path.join("advanced_testers", "advanced_tester_python_ctypes.py")],
        "log_file": "advanced_tester_python_ctypes.log"
    },
    {
        "name": "Fortran (Original)",
        "command": [os.path.join("bin", "advanced_tester_original.exe")],
        "log_file": "advanced_tester_fortran_original.log"
    },
    {
        "name": "Fortran (Refactored)",
        "command": [os.path.join("bin", "advanced_tester_refactored.exe")],
        "log_file": "advanced_tester_fortran_refactored.log"
    },
]

LOG_DIRECTORY = "logs"

# --- Helper Functions ---

def run_single_test(target_name: str, command: list[str], log_path: str):
    """
    Executes a single tester, captures its output, and returns the status.
    """
    print(f"  Running: {target_name}...")

    # Check if the executable or script exists before running
    executable = command[0]
    if not os.path.exists(executable):
        print(f"    -> ERROR: File not found: '{executable}'. Did you build it?")
        return "FAIL", f"File not found: {executable}"

    try:
        # Execute the command. The 'text=True' argument ensures stdout/stderr are strings.
        # 'encoding' and 'errors' arguments handle potential cross-platform console encoding issues.
        result = subprocess.run(
            command,
            capture_output=True,
            text=True,
            check=True,  # This will raise a CalledProcessError if the exit code is non-zero
            encoding='utf-8',
            errors='ignore'
        )

        # Write the standard output to the specified log file
        with open(log_path, 'w', encoding='utf-8') as f:
            f.write(result.stdout)

        # If we get here, the process ran successfully (exit code 0)
        print(f"    -> SUCCESS. Output saved to '{log_path}'")
        return "PASS", ""

    except FileNotFoundError:
        # This handles cases where the command itself isn't found (e.g., 'python' on a system without it)
        error_message = f"Command not found. Is '{command[0]}' in your system's PATH?"
        print(f"    -> ERROR: {error_message}")
        return "FAIL", error_message

    except subprocess.CalledProcessError as e:
        # This error is raised when the script runs but returns a non-zero exit code.
        # This typically indicates a failure within the tester itself.
        error_output = e.stdout + e.stderr
        with open(log_path, 'w', encoding='utf-8') as f:
            f.write("--- ERROR: Process returned a non-zero exit code. ---\n\n")
            f.write(error_output)

        print(f"    -> ERROR: Process failed with exit code {e.returncode}. Details in '{log_path}'")
        return "FAIL", f"Exit code {e.returncode}"
        
    except Exception as e:
        # Catch any other unexpected exceptions during the run.
        print(f"    -> UNEXPECTED ERROR: {e}")
        return "FAIL", str(e)


def print_summary_table(results: list[dict]):
    """
    Prints a formatted ASCII table of the test results.
    """
    # Define colors for PASS and FAIL for better visual feedback
    color_pass = '\033[92m'  # Green
    color_fail = '\033[91m'  # Red
    color_end = '\033[0m'     # Reset color

    # Disable colors on Windows Command Prompt as it doesn't support them by default
    is_windows_cmd = platform.system() == "Windows" and os.environ.get("TERM") is None

    print("\n" + "=" * 60)
    print("--- Advanced Tester Summary ---".center(60))
    print("=" * 60)
    print(f"{'Test Target':<30} | {'Status':<10} | {'Details'}")
    print("-" * 60)

    all_passed = True
    for res in results:
        status_str = res['status']
        if status_str == 'PASS':
            # Apply green color to PASS status
            status_display = f"{color_pass}{status_str}{color_end}" if not is_windows_cmd else status_str
        else:
            # Apply red color to FAIL status
            status_display = f"{color_fail}{status_str}{color_end}" if not is_windows_cmd else status_str
            all_passed = False
            
        print(f"{res['name']:<30} | {status_display:<18} | {res['details']}")

    print("-" * 60)
    if all_passed:
        print(f"{color_pass}All tests passed successfully!{color_end}" if not is_windows_cmd else "All tests passed successfully!")
    else:
        print(f"{color_fail}One or more tests failed. Please check the log files in the '{LOG_DIRECTORY}' directory.{color_end}" if not is_windows_cmd else f"One or more tests failed. Please check the log files in the '{LOG_DIRECTORY}' directory.")
    print("=" * 60)


# --- Main Execution ---

def main():
    """
    Main script logic to set up the environment and run all tests.
    """
    # Ensure the log directory exists
    if not os.path.exists(LOG_DIRECTORY):
        os.makedirs(LOG_DIRECTORY)

    # Check for PYTHONPATH, as it's required for the Python scripts to find modules
    if "PYTHONPATH" not in os.environ:
        print(f"Warning: PYTHONPATH environment variable is not set.", file=sys.stderr)
        print("Please run the 'setup_pythonpath.ps1' script before executing the tests.", file=sys.stderr)
        print("-" * 20)
    
    # Add a specific check for the MSYS_PYTHON executable
    if not os.path.exists(MSYS_PYTHON):
        print(f"Warning: The specified MSYS Python executable for f2py was not found at:", file=sys.stderr)
        print(f"  '{MSYS_PYTHON}'", file=sys.stderr)
        print("The 'Python (f2py)' test will likely fail. Please check the MSYS_PYTHON path in this script.", file=sys.stderr)
        print("-" * 20)


    print("Starting advanced correctness tests...")
    
    results = []
    for target in TESTER_TARGETS:
        log_path = os.path.join(LOG_DIRECTORY, target["log_file"])
        status, details = run_single_test(target["name"], target["command"], log_path)
        results.append({"name": target["name"], "status": status, "details": details})

    print_summary_table(results)

    # Exit with a non-zero status code if any test failed, useful for CI/CD pipelines
    if any(r['status'] == 'FAIL' for r in results):
        sys.exit(1)


if __name__ == "__main__":
    main()
