import numpy as np
import numba

@numba.njit
def drot1(n, dx, dy, c, s):
    # dx: ! 0ベース配列
    # dy: ! 0ベース配列

    if n <= 0:
        return

    # ! ループ制御を0ベース化 (0 から n-1 まで)
    for i in range(n):
        dtemp = c * dx[i] + s * dy[i]
        dy[i] = c * dy[i] - s * dx[i]
        dx[i] = dtemp

@numba.njit
def dswap1(n, dx, dy):

    if n <= 0:
        return

    #      CODE FOR BOTH INCREMENTS EQUAL TO 1
    #      CLEAN-UP LOOP

    m = n % 3 # Fortranの MOD(n,3) と同等
    if m != 0:
        for i in range(m): 
            dtemp = dx[i]
            dx[i] = dy[i]
            dy[i] = dtemp
        if n < 3:
            return
    
    # 2番目のループの開始インデックスは0ベースで m となる
    # ループ制御を0ベース化 (m から n-1 まで、ステップ3)
    for i in range(m, n, 3): 
        dtemp = dx[i]
        dx[i] = dy[i]
        dy[i] = dtemp
        
        dtemp = dx[i + 1]
        dx[i + 1] = dy[i + 1]
        dy[i + 1] = dtemp
        
        dtemp = dx[i + 2]
        dx[i + 2] = dy[i + 2]
        dy[i + 2] = dtemp


import math # math.sqrt を使用するため (ファイルの先頭に一度だけ記述)

# !      DESIGNED BY C.L.LAWSON, JPL, 1977 SEPT 08
# !
# !      CONSTRUCT THE GIVENS TRANSFORMATION
# !
# !               ( DC  DS )
# !         G = (          ) ,    DC**2 + DS**2 = 1 ,
# !               (-DS  DC )
# !
# !      WHICH ZEROS THE SECOND ENTRY OF THE 2-VECTOR  (DA,DB)**T .
# !
# !      THE QUANTITY R = (+/-)SQRT(DA**2 + DB**2) OVERWRITES DA IN
# !      STORAGE.  THE VALUE OF DB IS OVERWRITTEN BY A VALUE Z WHICH
# !      ALLOWS DC AND DS TO BE RECOVERED BY THE FOLLOWING ALGORITHM:
# !               IF Z=1  SET  DC=0.D0  AND  DS=1.D0
# !               IF DABS(Z) < 1  SET  DC=SQRT(1-Z**2)  AND  DS=Z
# !               IF DABS(Z) > 1  SET  DC=1/Z  AND  DS=SQRT(1-DC**2)
# !
# !      NORMALLY, THE SUBPROGRAM DROT(N,DX,INCX,DY,INCY,DC,DS) WILL
# !      NEXT BE CALLED TO APPLY THE TRANSFORMATION TO A 2 BY N MATRIX.
# ! ------------------------------------------------------------------
@numba.njit
def drotg(da_in, db_in):
    # FortranのINTENT(IN OUT)の動作を模倣するため、入力値をローカル変数にコピー
    da = float(da_in)
    db = float(db_in)
    # INTENT(OUT)の変数を初期化
    dc = 0.0
    ds = 0.0
    
    # REAL (dp)  :: u, v, r (Pythonでは事前の型宣言は不要)

    if abs(da) > abs(db):
        # ! *** HERE ABS(DA) > ABS(DB) ***
        u = da + da
        v = db / u
        # !     NOTE THAT U AND R HAVE THE SIGN OF DA
        r_val = math.sqrt(0.25 + v**2) * u
        # !     NOTE THAT DC IS POSITIVE
        dc = da / r_val # この時点のda (入力da) を使用
        ds = v * (dc + dc)
        db = ds # dbが更新される
        da = r_val # daが更新される
    elif db != 0.0:
        # ! *** HERE ABS(DA) <= ABS(DB) AND DB /= 0.D0 ***
        u = db + db
        v = da / u
        # !     NOTE THAT U AND R HAVE THE SIGN OF DB
        # !     (R IS IMMEDIATELY STORED IN DA)
        r_val_for_da = math.sqrt(0.25 + v**2) * u
        
        # Fortranでは da が先に更新され、その新しい da を使って ds が計算される
        original_db_val = db # このブロックでのds計算のための元のdb値
        da = r_val_for_da # daが更新される
        
        # !     NOTE THAT DS IS POSITIVE
        ds = original_db_val / da # 更新されたdaと、このブロックの開始時のdbでdsを計算
        dc = v * (ds + ds)
        
        if dc == 0.0:
            db = 1.0 # dbが更新される
        else:
            db = 1.0 / dc # dbが更新される
    else:
        # ! *** HERE DA = DB = 0.D0 ***
        dc = 1.0
        ds = 0.0
        # da, db は入力時の0.0のまま (変更なし)
        
    return da, db, dc, ds

@numba.njit
def dsvdc(x, n, p, s, e, u, v, job):
    # x: INTENT(IN OUT) NumPy array (0-based, x(0:n-1, 0:p-1))
    # n: INTENT(IN) integer
    # p: INTENT(IN) integer
    # s: INTENT(OUT) NumPy array (0-based, s(0:min(n,p)-1) effectively, allocated as s(0:MAX(n,p)-1) or similar by caller)
    # e: INTENT(OUT) NumPy array (0-based, e(0:p-1) effectively)
    # u: INTENT(OUT) NumPy array (0-based, u(0:n-1, 0:k-1))
    # v: INTENT(OUT) NumPy array (0-based, v(0:p-1, 0:p-1))
    # job: INTENT(IN) integer
    # info: INTENT(OUT) integer (returned by function)

    # dp precision is implicitly handled by Python floats / NumPy float64

    # INTERNAL VARIABLES
    # INTEGER :: iter, j, jobu, k, kase, kk, l, ll, lm1, lp1, ls, lu, m, maxit
    # INTEGER :: mm, mm1, nct, nctp1, ncu, nrt, nrtp1
    # REAL (dp) :: t
    # REAL (dp) :: b, c, cs, el, emm1, f, g, scale, shift, sl, sm, sn
    # REAL (dp) :: smm1, t1, test, ztest
    # LOGICAL :: wantu, wantv
    
    work = np.zeros(n, dtype=float) # REAL (dp) :: work(0:n-1)

    # SET THE MAXIMUM NUMBER OF ITERATIONS.
    maxit = 30

    # DETERMINE WHAT IS TO BE COMPUTED.
    wantu = False
    wantv = False
    jobu = (job % 100) // 10 # Integer division
    ncu = n
    if jobu > 1:
        ncu = min(n, p)
    if jobu != 0:
        wantu = True
    if (job % 10) != 0:
        wantv = True

    # REDUCE X TO BIDIAGONAL FORM, STORING THE DIAGONAL ELEMENTS
    # IN S AND THE SUPER-DIAGONAL ELEMENTS IN E.
    info = 0
    nct = min(n - 1, p) 
    # s(0:nct) = 0.0_dp -> s is 0-based, nct is max 0-based index
    # So, if nct = 2 (indices 0,1,2), elements s[0],s[1],s[2] are set.
    # Python slice s[0:nct+1]
    if nct >= 0 : # Ensure slice is valid if nct is -1 (e.g. n=0)
         s[0:nct+1] = 0.0 

    nrt = max(0, min(p - 2, n))
    lu = max(nct, nrt)
    
    if lu >= 1: # Fortran DO L = 1, LU
        for l_loop_var in range(1, lu + 1):
            lp1 = l_loop_var + 1
            if l_loop_var <= nct:
                # s(l-1) = SQRT( SUM( x(l-1:n-1,l-1)**2 ) )
                # x column slice: x[l_loop_var-1:n, l_loop_var-1]
                s[l_loop_var-1] = math.sqrt(np.sum(x[l_loop_var-1:n, l_loop_var-1]**2))
                
                if s[l_loop_var-1] != 0.0:
                    if x[l_loop_var-1, l_loop_var-1] != 0.0:
                        s[l_loop_var-1] = np.copysign(s[l_loop_var-1], x[l_loop_var-1, l_loop_var-1])
                    x[l_loop_var-1:n, l_loop_var-1] = x[l_loop_var-1:n, l_loop_var-1] / s[l_loop_var-1]
                    x[l_loop_var-1, l_loop_var-1] = 1.0 + x[l_loop_var-1, l_loop_var-1]
                s[l_loop_var-1] = -s[l_loop_var-1]
            
            if p >= lp1: # Fortran DO J = LP1, P
                for j_loop_var in range(lp1, p + 1):
                    if l_loop_var <= nct and s[l_loop_var-1] != 0.0:
                        # APPLY THE TRANSFORMATION.
                        # t = -DOT_PRODUCT(x(l-1:n-1,l-1), x(l-1:n-1,j-1)) / x(l-1,l-1)
                        t = -np.dot(x[l_loop_var-1:n, l_loop_var-1], x[l_loop_var-1:n, j_loop_var-1]) / x[l_loop_var-1, l_loop_var-1]
                        x[l_loop_var-1:n, j_loop_var-1] = x[l_loop_var-1:n, j_loop_var-1] + t * x[l_loop_var-1:n, l_loop_var-1]
                    
                    # PLACE THE L-TH ROW OF X INTO E FOR THE
                    # SUBSEQUENT CALCULATION OF THE ROW TRANSFORMATION.
                    e[j_loop_var-1] = x[l_loop_var-1, j_loop_var-1]
            
            if wantu and l_loop_var <= nct:
                # PLACE THE TRANSFORMATION IN U FOR SUBSEQUENT BACK MULTIPLICATION.
                # u(l-1 : n-1, l-1) = x(l-1:n-1,l-1)
                u[l_loop_var-1:n, l_loop_var-1] = x[l_loop_var-1:n, l_loop_var-1]

            if l_loop_var > nrt:
                continue # CYCLE
            
            # COMPUTE THE L-TH ROW TRANSFORMATION AND PLACE THE
            # L-TH SUPER-DIAGONAL IN e(l-1).
            # e(l-1) = SQRT( SUM( e(lp1-1:p-1)**2 ) )
            # Slice e[lp1-1:p]
            e[l_loop_var-1] = math.sqrt(np.sum(e[lp1-1:p]**2))
            
            if e[l_loop_var-1] != 0.0: # GOTO 80 の除去：IF条件を反転し、ブロック化
                if e[lp1-1] != 0.0:
                    e[l_loop_var-1] = np.copysign(e[l_loop_var-1], e[lp1-1])
                # e(lp1-1:lp1+p-l-2) = e(lp1-1:p-1) / e(l-1)
                # LHS slice e[lp1-1 : lp1+p-l_loop_var-1]
                # RHS slice e[lp1-1 : p]
                e[lp1-1 : lp1+p-l_loop_var-1] = e[lp1-1 : p] / e[l_loop_var-1]
                e[lp1-1] = 1.0 + e[lp1-1]
            
            e[l_loop_var-1] = -e[l_loop_var-1]
            
            if lp1 <= n and e[l_loop_var-1] != 0.0:
                # APPLY THE TRANSFORMATION.
                # work(lp1-1 : n-1) = 0.0D0
                work[lp1-1:n] = 0.0
                # DO j = lp1, p
                for j_loop_var in range(lp1, p + 1):
                    # work(lp1-1 : lp1+n-l-2) = work(lp1-1 : lp1+n-l-2) + e(j-1) * x(lp1-1 : lp1+n-l-2, j-1)
                    # work slice: work[lp1-1 : lp1+n-l_loop_var-1]
                    # x slice: x[lp1-1 : lp1+n-l_loop_var-1, j_loop_var-1]
                    slice_len_n_minus_l = n - l_loop_var # num elements in Fortran slice (lp1+n-l-2) - (lp1-1) + 1
                    if slice_len_n_minus_l > 0: # ensure slice is valid
                        work[lp1-1 : lp1-1+slice_len_n_minus_l] = work[lp1-1 : lp1-1+slice_len_n_minus_l] + \
                            e[j_loop_var-1] * x[lp1-1 : lp1-1+slice_len_n_minus_l, j_loop_var-1]

                # DO j = lp1, p
                for j_loop_var in range(lp1, p + 1):
                    # x(lp1-1 : lp1+n-l-2, j-1) = x(lp1-1 : lp1+n-l-2, j-1) - (e(j-1)/e(lp1-1)) * work(lp1-1 : lp1+n-l-2)
                    if slice_len_n_minus_l > 0: # ensure slice is valid
                        x[lp1-1 : lp1-1+slice_len_n_minus_l, j_loop_var-1] = x[lp1-1 : lp1-1+slice_len_n_minus_l, j_loop_var-1] - \
                            (e[j_loop_var-1]/e[lp1-1]) * work[lp1-1 : lp1-1+slice_len_n_minus_l]
            
            if not wantv:
                continue # CYCLE
            
            # PLACE THE TRANSFORMATION IN V FOR SUBSEQUENT
            # BACK MULTIPLICATION.
            # v(lp1-1 : p-1, l-1) = e(lp1-1:p-1)
            v[lp1-1:p, l_loop_var-1] = e[lp1-1:p]

    # SET UP THE FINAL BIDIAGONAL MATRIX OF ORDER M.
    m = min(p, n + 1)
    nctp1 = nct + 1
    nrtp1 = nrt + 1
    
    if nct < p:
        s[nctp1-1] = x[nctp1-1, nctp1-1]
    if n < m:
        s[m-1] = 0.0
    if nrtp1 < m:
        # Fortran index m-1 is 0-based index for column of x, so x[..., m-1]
        e[nrtp1-1] = x[nrtp1-1, m-1] 
    
    # Ensure e is 0-indexed according to problem statement (e(0:p-1))
    # The original code e(m-1) might go out of bounds if m-1 >= p for e(0:p-1)
    # However, m = min(p, n+1). If m=p+1 (when n+1 < p), then e(p) is accessed.
    # LINPACK e is typically P-1 in size (e(0:P-2) effectively for superdiag of PxP bidiag).
    # Given Fortran code: E is P-elements (0:P-1).
    # s is MIN(N+1,P) elements.
    # For e(m-1) = 0.0D0. Max m-1 is p. So e[p] if m=p+1.
    # This line assumes e has at least m elements. (e.g., size MAX(P, MIN(P,N+1)) )
    # If e is size P, then m-1 must be < P.
    # m = min(p, n+1). If n+1 <= p, then m = n+1. e(n) = 0. OK if n < p.
    # If p < n+1, then m = p. e(p-1) = 0. OK.
    if m-1 < len(e): # Protect access
        e[m-1] = 0.0

    # IF REQUIRED, GENERATE U.
    if wantu:
        if ncu >= nctp1: # GOTO 200 の除去：IF条件を反転し、ブロック化
            for j_loop_var in range(nctp1, ncu + 1): # Fortran DO J = NCTP1, NCU
                u[0:n, j_loop_var-1] = 0.0
                u[j_loop_var-1, j_loop_var-1] = 1.0
        
        # ラベル 200 は除去されました
        if nct >= 1: # Fortran DO LL = 1, NCT
            for ll_loop_var in range(1, nct + 1):
                l_val = nct - ll_loop_var + 1 # l in Fortran
                if s[l_val-1] != 0.0:
                    lp1 = l_val + 1
                    if ncu >= lp1:
                        for j_loop_var in range(lp1, ncu + 1): # Fortran DO J = LP1, NCU
                            # t = -DOT_PRODUCT(u(l-1:n-1,l-1), u(l-1:n-1,j-1)) / u(l-1,l-1)
                            t = -np.dot(u[l_val-1:n, l_val-1], u[l_val-1:n, j_loop_var-1]) / u[l_val-1, l_val-1]
                            u[l_val-1:n, j_loop_var-1] = u[l_val-1:n, j_loop_var-1] + t * u[l_val-1:n, l_val-1]
                    
                    u[l_val-1:n, l_val-1] = -u[l_val-1:n, l_val-1]
                    u[l_val-1, l_val-1] = 1.0 + u[l_val-1, l_val-1]
                    lm1 = l_val - 1
                    if lm1 >= 1:
                        # u(0:lm1-1,l-1) = 0.0_dp -> u[0:lm1, l_val-1]
                        u[0:lm1, l_val-1] = 0.0
                else:
                    u[0:n, l_val-1] = 0.0
                    u[l_val-1, l_val-1] = 1.0
    
    # IF IT IS REQUIRED, GENERATE V.
    # ラベル 300 は除去されました
    if wantv:
        if p >=1: # Fortran DO LL = 1, P
            for ll_loop_var in range(1, p + 1):
                l_val = p - ll_loop_var + 1 # l in Fortran
                lp1 = l_val + 1
                if l_val <= nrt and e[l_val-1] != 0.0:
                    if lp1 <= p: # Ensure loop j=lp1,p is valid
                        for j_loop_var in range(lp1, p + 1): # Fortran DO J = LP1, P
                            # t = -DOT_PRODUCT(v(lp1-1 : lp1+p-l-1-1, l-1), v(lp1-1 : lp1+p-l-1-1, j-1)) / v(lp1-1, l-1)
                            # Slice indices: lp1-1 to lp1+p-l_val-2
                            # Python slice: v[lp1-1 : lp1+p-l_val-1, l_val-1]
                            idx_start_v_slice = lp1-1
                            idx_end_v_slice = lp1+p-l_val-1 
                            if idx_start_v_slice < idx_end_v_slice : # ensure slice is valid
                                t = -np.dot(v[idx_start_v_slice:idx_end_v_slice, l_val-1], 
                                            v[idx_start_v_slice:idx_end_v_slice, j_loop_var-1]) / v[lp1-1, l_val-1]
                                v[idx_start_v_slice:idx_end_v_slice, j_loop_var-1] = \
                                    v[idx_start_v_slice:idx_end_v_slice, j_loop_var-1] + \
                                    t * v[idx_start_v_slice:idx_end_v_slice, l_val-1]
                
                v[0:p, l_val-1] = 0.0
                v[l_val-1, l_val-1] = 1.0

    # MAIN ITERATION LOOP FOR THE SINGULAR VALUES.
    mm = m # mm is m from before this main loop
    iter_count = 0 # iter in Fortran
    
    # Fortran DO ... END DO translates to while True with break conditions
    while True:
        # QUIT IF ALL THE SINGULAR VALUES HAVE BEEN FOUND.
        if m == 0:
            break

        # IF TOO MANY ITERATIONS HAVE BEEN PERFORMED, SET FLAG AND RETURN.
        if iter_count >= maxit:
            info = m
            break
        
        # THIS SECTION OF THE PROGRAM INSPECTS FOR NEGLIGIBLE ELEMENTS
        # IN THE S AND E ARRAYS.
        l_val = 0 # Default for l if loop doesn't run or condition never met
        # Fortran DO LL = 1, M (l = m - ll) is equivalent to l from m-1 down to 0
        # The EXIT condition is l==0 (which means ll=m)
        # So loop for l from m-1 down to 0.
        # The loop variable l_in_ll_loop is the l from the Fortran code: l = m-ll
        # Fortran: DO ll=1,m -> l = m-1, m-2, ..., 0
        # Python: for l_in_ll_loop in range(m - 1, -1, -1):
        # The original code uses l = m-ll, and exits when l == 0.
        # The l is then used *after* this loop.
        
        # Fortran:
        # DO ll = 1, m
        #   l = m - ll
        #   IF (l == 0) EXIT ! l will be 0 here, then used
        #   ...
        # END DO
        # This sets l. l=0 if loop completes fully, or l is value when test broke.
        
        found_l_in_first_test = False
        for ll_loop_var_main_iter in range(1, m + 1):
            l_val = m - ll_loop_var_main_iter # l in Fortran
            if l_val == 0: # Corresponds to Fortran IF (l == 0) EXIT
                found_l_in_first_test = True
                break 
            # Fortran indices: s(l-1) and s(l) are s[l_val-1] and s[l_val]
            # e(l-1) is e[l_val-1]
            test = abs(s[l_val-1]) + abs(s[l_val])
            ztest = test + abs(e[l_val-1])
            if ztest == test:
                e[l_val-1] = 0.0
                found_l_in_first_test = True
                break # Exit DO ll loop
        if not found_l_in_first_test and m > 0: # if loop finished without break, l_val is 0
             l_val = 0 # as per Fortran when loop terminates naturally after ll=m

        kase = 0 # Initialize kase
        if l_val == m - 1:
            kase = 4
            l_val = l_val + 1 # l_val becomes m
        else:
            # ls の値を計算する修正済みのループ
            ls_val = l_val # lsの初期値を設定 (ここでの 'l_val' は最初のe(l_val)検査の値)
            # Fortran: DO k = m, l + 1, -1
            # Python: range(m, (l_val + 1) - 1, -1) which is range(m, l_val, -1)
            found_ls = False
            for k_loop_var in range(m, l_val, -1): # k from m down to l_val+1
                test = 0.0
                if k_loop_var != m: # Fortran k /= m
                    test = test + abs(e[k_loop_var-1])
                if k_loop_var != l_val + 1: # Fortran k /= l + 1
                    test = test + abs(e[k_loop_var-1-1]) # e[k_loop_var-2]
                
                ztest = test + abs(s[k_loop_var-1])
                if ztest == test:
                    s[k_loop_var-1] = 0.0
                    ls_val = k_loop_var
                    found_ls = True
                    break # EXIT from DO k loop
            # ls が設定された後。ここでの 'l_val' は依然として最初のe(l_val)検査の値を保持
            
            if ls_val == l_val:
                kase = 3
                l_val = l_val + 1
            elif ls_val == m:
                kase = 1
                l_val = l_val + 1
            else: # ls_val is somewhere between l_val and m
                kase = 2
                l_val = ls_val + 1 # kase=2 の場合は ls を基に l を更新

        # PERFORM THE TASK INDICATED BY KASE.
        if kase == 1:
            # DEFLATE NEGLIGIBLE s(m-1).
            mm1 = m - 1
            f = e[m-1-1] # e[m-2]
            e[m-1-1] = 0.0
            # Fortran: DO kk = l, mm1 -> k = mm1 - kk + l
            # This makes k go from mm1 down to l_val
            for kk_loop_var in range(l_val, mm1 + 1):
                k_val = mm1 - kk_loop_var + l_val
                
                t1_in = s[k_val-1]
                f_in = f
                t1_out, f_out, cs, sn = drotg(t1_in, f_in)
                s[k_val-1] = t1_out
                f = f_out
                
                if k_val != l_val:
                    f = -sn * e[k_val-1-1] # e[k_val-2]
                    e[k_val-1-1] = cs * e[k_val-1-1]
                
                if wantv:
                    # drot1(p, v(0:,k-1), v(0:,m-1), cs, sn)
                    drot1(p, v[:, k_val-1], v[:, m-1], cs, sn)
            continue # CYCLE to next main iteration

        elif kase == 2:
            # SPLIT AT NEGLIGIBLE s(l-1).
            f = e[l_val-1-1] # e[l_val-2]
            e[l_val-1-1] = 0.0
            for k_loop_var in range(l_val, m + 1): # Fortran DO k = l, m
                t1_in = s[k_loop_var-1]
                f_in = f
                t1_out, f_out, cs, sn = drotg(t1_in, f_in)
                s[k_loop_var-1] = t1_out
                f = f_out
                
                f = -sn * e[k_loop_var-1]
                e[k_loop_var-1] = cs * e[k_loop_var-1]
                
                if wantu:
                    # drot1(n, u(0:,k-1), u(0:,l-1-1), cs, sn)
                    # u[:,l_val-2]
                    drot1(n, u[:, k_loop_var-1], u[:, l_val-1-1], cs, sn)
            continue # CYCLE

        elif kase == 3:
            # PERFORM ONE QR STEP.
            # CALCULATE THE SHIFT.
            # scale = MAX(ABS(s(m-1)), ABS(s(m-2)), ABS(e(m-1-1)), ABS(s(l-1)), ABS(e(l-1)))
            scale = max(abs(s[m-1]), abs(s[m-2]), abs(e[m-1-1]), abs(s[l_val-1]), abs(e[l_val-1]))
            
            sm = s[m-1]/scale
            smm1 = s[m-2]/scale
            emm1 = e[m-1-1]/scale # e[m-2]
            sl = s[l_val-1]/scale
            el = e[l_val-1]/scale
            
            b = ((smm1 + sm)*(smm1 - sm) + emm1**2)/2.0
            c_val = (sm*emm1)**2 # c is a keyword in Python often for char
            shift = 0.0
            if b != 0.0 or c_val != 0.0:
                shift = math.sqrt(b**2 + c_val)
                if b < 0.0:
                    shift = -shift
                shift = c_val/(b + shift)
            
            f = (sl + sm)*(sl - sm) - shift
            g = sl*el
            
            # CHASE ZEROS.
            mm1 = m - 1
            for k_loop_var in range(l_val, mm1 + 1): # Fortran DO k = l, mm1
                f_in = f
                g_in = g
                f, g, cs, sn = drotg(f_in, g_in)
                
                if k_loop_var != l_val:
                    e[k_loop_var-1-1] = f # e[k_loop_var-2]
                
                f = cs*s[k_loop_var-1] + sn*e[k_loop_var-1]
                e[k_loop_var-1] = cs*e[k_loop_var-1] - sn*s[k_loop_var-1]
                g = sn*s[k_loop_var] # s[k]
                s[k_loop_var] = cs*s[k_loop_var]
                
                if wantv:
                    # drot1(p, v(0:,k-1), v(0:,k), cs, sn)
                    drot1(p, v[:, k_loop_var-1], v[:, k_loop_var], cs, sn)

                f_in = f
                g_in = g
                f, g, cs, sn = drotg(f_in, g_in)
                
                s[k_loop_var-1] = f
                f = cs*e[k_loop_var-1] + sn*s[k_loop_var]
                s[k_loop_var] = -sn*e[k_loop_var-1] + cs*s[k_loop_var]
                g = sn*e[k_loop_var] # e[k]
                e[k_loop_var] = cs*e[k_loop_var]
                
                if wantu and k_loop_var < n : # Fortran k < n
                     # drot1(n, u(0:,k-1), u(0:,k), cs, sn)
                    drot1(n, u[:, k_loop_var-1], u[:, k_loop_var], cs, sn)
            
            e[m-1-1] = f # e[m-2]
            iter_count = iter_count + 1
            continue # CYCLE
            
        elif kase == 4:
            # CONVERGENCE.
            # MAKE THE SINGULAR VALUE POSITIVE.
            if s[l_val-1] < 0.0:
                s[l_val-1] = -s[l_val-1]
                if wantv:
                    v[0:p, l_val-1] = -v[0:p, l_val-1]
            
            # ORDER THE SINGULAR VALUE.
            # Fortran DO ... END DO loop (label 590)
            while True:
                if l_val == mm: # Condition 1 for exit
                    break
                # s[l_val] is s(l) in Fortran. s[l_val-1] is s(l-1)
                if s[l_val-1] >= s[l_val]: # Condition 2 for exit (l < mm and order is correct)
                    break
                
                # If conditions not met (l < mm AND s(l-1) < s(l))
                t = s[l_val-1]
                s[l_val-1] = s[l_val]
                s[l_val] = t
                
                if wantv and l_val < p: # Fortran l < p
                    # dswap1(p, v(0:,l-1), v(0:,l))
                    dswap1(p, v[:, l_val-1], v[:, l_val])
                if wantu and l_val < n: # Fortran l < n
                    # dswap1(n, u(0:,l-1), u(0:,l))
                    dswap1(n, u[:, l_val-1], u[:, l_val])
                l_val = l_val + 1
            
            iter_count = 0 # Fortran iter = 0
            m = m - 1    # Fortran m = m - 1
            continue # CYCLE
            
    return info
