# build-benchmarks.ps1 (Final Version with independent ctypes build)

<#
.SYNOPSIS
    Builds and cleans the benchmark executables and DLLs for the Fortran project.
.DESCRIPTION
    By default, this script builds the benchmark executables and the ctypes DLL.
    It places intermediate files (.o, .mod) in ./build/
    It places final artifacts (.exe, .dll) in ./bin/
    Using the -Clean switch will remove the generated build artifacts.
.EXAMPLE
    # To build the project
    ./build-benchmarks.ps1
.EXAMPLE
    # To clean up generated files
    ./build-benchmarks.ps1 -Clean
#>
param (
    [Switch]$Clean
)

$ErrorActionPreference = "Stop"

# --- Configuration ---
$Compiler = "gfortran"
$CompilerFlags = @("-O3", "-Wall")

# --- Directory Paths ---
$BenchmarkDir = "benchmarks"
$SourceDirOriginal = "fortran_src/original"
$SourceDirRefactored = "fortran_src/refactored"
$SourceDirCtypes = "fortran_src/ctypes"
$BuildDir = "build" # For intermediate files (.o, .mod)
$BinDir = "bin"     # For final executables (.exe)

# --- Source File Definitions ---
$SourceFilesOriginal = @(
    (Join-Path -Path $SourceDirOriginal -ChildPath "dsvdc.f90"),
    (Join-Path -Path $BenchmarkDir -ChildPath "benchmark_svd_fortran_original.f90")
)
$SourceFilesRefactored = @(
    (Join-Path -Path $SourceDirRefactored -ChildPath "dsvdc.f90"),
    (Join-Path -Path $BenchmarkDir -ChildPath "benchmark_svd_fortran_refactored.f90")
)
$SourceFilesCtypes = @( # <-- 修正: ctypes用のソースを配列で定義
    (Join-Path -Path $SourceDirCtypes -ChildPath "dsvdc.f90"),
    (Join-Path -Path $SourceDirCtypes -ChildPath "dsvdc_wrapper.f90")
)

# --- Target Definitions ---
$TargetOriginal = Join-Path -Path $BinDir -ChildPath "benchmark_svd_fortran_original.exe"
$TargetRefactored = Join-Path -Path $BinDir -ChildPath "benchmark_svd_fortran_refactored.exe"
$TargetCtypesDll = Join-Path -Path $BinDir -ChildPath "svd_ctypes.dll"

# --- Main Logic ---
if ($Clean) {
    Write-Host "--- Cleaning up project files... ---"
    if (Test-Path $BuildDir) {
        Write-Host "Removing directory $BuildDir..."
        Remove-Item -Path $BuildDir -Recurse -Force
    }
    if (Test-Path $BinDir) {
        Write-Host "Removing directory $BinDir..."
        Remove-Item -Path $BinDir -Recurse -Force
    }
    Write-Host "Cleanup complete."
} else {
    # --- Create Directories ---
    if (-not (Test-Path $BuildDir)) { New-Item -ItemType Directory -Path $BuildDir | Out-Null }
    if (-not (Test-Path $BinDir)) { New-Item -ItemType Directory -Path $BinDir | Out-Null }

    # --- Intermediate File Definitions ---
    $ObjectFileOriginal = Join-Path -Path $BuildDir -ChildPath "dsvdc_original.o"
    $ObjectFileRefactored = Join-Path -Path $BuildDir -ChildPath "dsvdc_refactored.o"
    $ObjectFileCtypes = Join-Path -Path $BuildDir -ChildPath "dsvdc_ctypes.o"

    # Add compiler flag to place .mod files into the build directory
    $CompilerFlagsWithMod = $CompilerFlags + "-J$BuildDir"

    # --- Build Original Version ---
    Write-Host "--- Building original version... ---"
    & $Compiler $CompilerFlagsWithMod -c $SourceFilesOriginal[0] -o $ObjectFileOriginal
    & $Compiler $CompilerFlagsWithMod $SourceFilesOriginal[1] $ObjectFileOriginal -o $TargetOriginal
    Write-Host "Successfully created $TargetOriginal"
    Write-Host ""

    # --- Build Refactored Version ---
    Write-Host "--- Building refactored version... ---"
    & $Compiler $CompilerFlagsWithMod -c $SourceFilesRefactored[0] -o $ObjectFileRefactored
    & $Compiler $CompilerFlagsWithMod $SourceFilesRefactored[1] $ObjectFileRefactored -o $TargetRefactored
    Write-Host "Successfully created $TargetRefactored"
    Write-Host ""

    # --- Build ctypes DLL (Independent) --- # <-- 修正ブロック
    Write-Host "--- Building ctypes DLL (Independent)... ---"
    # 1. ctypes版のdsvdc.f90をコンパイルし、専用のオブジェクトファイルを作成
    Write-Host "Compiling module: $($SourceFilesCtypes[0])"
    & $Compiler $CompilerFlagsWithMod -c $SourceFilesCtypes[0] -o $ObjectFileCtypes
    if ($LASTEXITCODE -ne 0) { Write-Error "Failed to compile ctypes module. Halting."; exit 1 }

    # 2. 上で作成したオブジェクトファイルを使い、ラッパーをコンパイルしてDLLを生成
    Write-Host "Compiling shared library: $($SourceFilesCtypes[1]) (linking with $ObjectFileCtypes)"
    $DllCompilerFlags = $CompilerFlagsWithMod + "-shared"
    & $Compiler $DllCompilerFlags $SourceFilesCtypes[1] $ObjectFileCtypes -o $TargetCtypesDll
    if ($LASTEXITCODE -ne 0) { Write-Error "Failed to build ctypes DLL. Halting."; exit 1 }
    Write-Host "Successfully created $TargetCtypesDll"
    Write-Host ""


    Write-Host "Build complete. Artifacts are in '$BinDir' directory."
}