<#
.SYNOPSIS
    Builds the Fortran-based advanced tester executables.
.DESCRIPTION
    This script compiles the 'original' and 'refactored' versions of the advanced
    correctness testers.
    - It uses the dsvdc.f90 source from the corresponding 'fortran_src' subdirectories.
    - Intermediate files (.o, .mod) are placed in ./build/
    - Final executables (.exe) are placed in ./bin/

    Use the -Clean switch to remove the build artifacts (cleans both benchmark and
    tester artifacts by removing the ./bin and ./build directories).
.EXAMPLE
    # To build the advanced tester executables
    ./build-advanced-testers.ps1

.EXAMPLE
    # To clean up all generated build files (for all scripts)
    ./build-advanced-testers.ps1 -Clean
#>
param (
    [Switch]$Clean
)

$ErrorActionPreference = "Stop"

# --- Configuration ---
$Compiler = "gfortran"
$CompilerFlags = @("-O3", "-Wall", "-fcheck=bounds") # Added bounds checking for testing

# --- Directory Paths ---
$AdvancedTesterDir = "advanced_testers"
$SourceDirOriginal = "fortran_src/original"
$SourceDirRefactored = "fortran_src/refactored"
$BuildDir = "build" # For intermediate files (.o, .mod)
$BinDir = "bin"     # For final executables (.exe)

# --- Source File Definitions ---
# Sources for the 'Original' version tester
$SourceFileDsvdcOriginal = Join-Path -Path $SourceDirOriginal -ChildPath "dsvdc.f90"
$SourceFileTesterOriginal = Join-Path -Path $AdvancedTesterDir -ChildPath "advanced_tester_original.f90"

# Sources for the 'Refactored' version tester
$SourceFileDsvdcRefactored = Join-Path -Path $SourceDirRefactored -ChildPath "dsvdc.f90"
$SourceFileTesterRefactored = Join-Path -Path $AdvancedTesterDir -ChildPath "advanced_tester_refactored.f90"


# --- Target Executable Definitions ---
$TargetTesterOriginal = Join-Path -Path $BinDir -ChildPath "advanced_tester_original.exe"
$TargetTesterRefactored = Join-Path -Path $BinDir -ChildPath "advanced_tester_refactored.exe"

# --- Main Logic ---
if ($Clean) {
    Write-Host "--- Cleaning up all project build files... ---" -ForegroundColor Yellow
    if (Test-Path $BuildDir) {
        Write-Host "Removing directory $BuildDir..."
        Remove-Item -Path $BuildDir -Recurse -Force
    }
    if (Test-Path $BinDir) {
        Write-Host "Removing directory $BinDir..."
        Remove-Item -Path $BinDir -Recurse -Force
    }
    Write-Host "Cleanup complete."
} else {
    # --- Create Directories if they don't exist ---
    if (-not (Test-Path $BuildDir)) { New-Item -ItemType Directory -Path $BuildDir | Out-Null }
    if (-not (Test-Path $BinDir)) { New-Item -ItemType Directory -Path $BinDir | Out-Null }

    # --- Intermediate Object File Definitions ---
    # These object files can be shared with the benchmark build script
    $ObjectFileDsvdcOriginal = Join-Path -Path $BuildDir -ChildPath "dsvdc_original.o"
    $ObjectFileDsvdcRefactored = Join-Path -Path $BuildDir -ChildPath "dsvdc_refactored.o"

    # Add compiler flag to place .mod files into the build directory
    $CompilerFlagsWithMod = $CompilerFlags + "-J$BuildDir"

    # --- Build Original Version Tester ---
    Write-Host "--- Building advanced tester (original version)... ---" -ForegroundColor Green
    # 1. Compile the 'original' dsvdc.f90 into an object file
    Write-Host "Compiling module: $SourceFileDsvdcOriginal"
    & $Compiler $CompilerFlagsWithMod -c $SourceFileDsvdcOriginal -o $ObjectFileDsvdcOriginal
    if ($LASTEXITCODE -ne 0) { Write-Error "Failed to compile original dsvdc module. Halting."; exit 1 }

    # 2. Compile and link the tester executable
    Write-Host "Compiling and linking executable: $SourceFileTesterOriginal"
    & $Compiler $CompilerFlagsWithMod $SourceFileTesterOriginal $ObjectFileDsvdcOriginal -o $TargetTesterOriginal
    if ($LASTEXITCODE -ne 0) { Write-Error "Failed to build original advanced tester. Halting."; exit 1 }
    Write-Host "Successfully created $TargetTesterOriginal"
    Write-Host ""

    # --- Build Refactored Version Tester ---
    Write-Host "--- Building advanced tester (refactored version)... ---" -ForegroundColor Green
    # 1. Compile the 'refactored' dsvdc.f90 into an object file
    Write-Host "Compiling module: $SourceFileDsvdcRefactored"
    & $Compiler $CompilerFlagsWithMod -c $SourceFileDsvdcRefactored -o $ObjectFileDsvdcRefactored
    if ($LASTEXITCODE -ne 0) { Write-Error "Failed to compile refactored dsvdc module. Halting."; exit 1 }

    # 2. Compile and link the tester executable
    Write-Host "Compiling and linking executable: $SourceFileTesterRefactored"
    & $Compiler $CompilerFlagsWithMod $SourceFileTesterRefactored $ObjectFileDsvdcRefactored -o $TargetTesterRefactored
    if ($LASTEXITCODE -ne 0) { Write-Error "Failed to build refactored advanced tester. Halting."; exit 1 }
    Write-Host "Successfully created $TargetTesterRefactored"
    Write-Host ""


    Write-Host "Build complete. Tester executables are in the '$BinDir' directory."
}
